/*
		Facility:
			SpatialObjects - Geographic Network Management
	        Copyright 2001 Celeritas Technologies, L.L.C.
			
	 	Title:
	   		ctValidateNumber.js
			
		Abstract:
			This file contains various functions that provide
			validation of numbers.  Most of these functions are intended
			to validate numbers only when the numbers are expressed as
			string literals.  See the comments associated with each
			individual function.
	      
	 	Author:
			Robert L. Marshall/Celeritas Technologies, LLC
	
	  	Creation Date:
	 		30-Apr-2001
	
		Modification History:
			dd-mmm-yyyy	(iii)

		See Also:
			ctValidateTest.html
	
		Dependencies:
			ctValidateCommon.js
*/

/*
	----------------------------------------------------------------------
	The test file for this file is ctValidateTest.html.  If any
	changes are made to the functions in this file, those changes must be
	tested by running the test file.
	----------------------------------------------------------------------
	
	pbEmptyOK:
	Some of the functions in this file provide the optional argument
	pbEmptyOK.  If a function provides argument pbEmptyOK, when a call to
	that function is made, if argument pbEmptyOK is not empty, pbEmptyOK
	(set to true or false) is used to set the function's return value
	when psValue is empty.  If pbEmptyOK is an empty argument, the
	global default variable	bNumberDefaultEmptyOK is used to set the
	function's return value.
	
	Global default variable bNumberDefaultEmptyOK is initialized in
	file ctValidateCommon.js.
	
	For the meaning of "empty," see the function isEmpty in file
	ctValidateCommon.js.
	
*/
		//
		// This function receives a string and returns true if it is an
		// unsigned floating point decimal number.  An unsigned floating
		// point decimal number is defined as follows:
		// 		
		//	(a)  A decimal number (not octal or hexadecimal).
		//	(b)  No leading "+" or "-" sign.
		//	(c)  One decimal point.
		//	(d)  No leading, intervening, or trailing spaces.
		//	(e)  One of the two following forms:
		//	     	(e1) one or more digits, followed by a decimal point,
		//			     followed by zero or more digits.
		//			(e2) zero or more digits, followed by a decimal point,
		//			     followed by one or more digits.
		//
		// A floating point number can contain one or more leading zeros.
		//
		// An integer (not containing a decimal point) is not considered
		// to be a floating point number.  A string in exponential
		// notation is not considered to be a floating point number.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isFloatUnsigned (psValue, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			//	
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {

				return (bNumberDefaultEmptyOK == true);
			}

		} else {

			var rRE = /^((\d+\.\d*)|(\d*\.\d+))$/;
	    	return rRE.test(psValue);
		}
	}



		// This function receives a string and returns true if it is an
		// unsigned floating point decimal number within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		// For the definition of an unsigned floating point decimal
		// number, see comments associated with the function
		// isFloatUnsigned.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isFloatUnsignedInRange (psValue, pnMinValue, pnMaxValue,
											pbExclusive, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			// This verification must be done here at the start of the
			// function, rather than via the call below to function
			// isFloatUnsigned, since pbEmptyOK or
			// bNumberDefaultEmptyOK could allow an empty psValue
			// to be acceptable, which could cause incorrect operation of
			// the limit checking logic below.
			//													
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {
	
				return (bNumberDefaultEmptyOK == true);
			}

		} else {
				//
				// Verify that the argument to be validated is a valid
				// unsigned floating point number.  pbEmptyOK does not
				// need to be passed to isFloatUnsigned, since checking
				// for an empty psValue was performed above.
				//	
			if (isFloatUnsigned (psValue)) {
					//
					// For the limit check, the default is to include
					// the minimum and maximum limits, rather than to
					// exclude them.
					//
				if (isEmpty(pbExclusive) || pbExclusive != true) {
						//
					    // Check minimum (inclusively)
						//
				    if (!isEmpty(pnMinValue)) {
				        if (parseFloat(psValue) < pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (inclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
					
						if (parseFloat(psValue) > pnMaxValue) {
							return false;
						}
					}
	
				    return true;
					
				} else {
						//
					    // Check minimum (exclusively)
						//
				    if (!isEmpty(pnMinValue)) {
		
				        if (parseFloat(psValue) <= pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (exclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
		
						if (parseFloat(psValue) >= pnMaxValue) {
							return false;
						}
					}
					
				    return true;			
				
				}		
				
			} else {
			
				return false;
				
			}
		}
	}

	

		// This function receives a string and returns true if it is a
		// signed floating point decimal number.  A signed floating
		// point decimal number is defined as follows:
		// 		
		//	(a)  Must contain a leading "+" or "-" sign, followed by (b).
		//	(b)  An unsigned floating point string, as defined in the
		//		 function isFloatUnsigned.
		//	(c)  No spaces between the sign and the unsigned floating
		//		 point string.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isFloatSigned (psValue, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			//	
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {

				return (bNumberDefaultEmptyOK == true);
			}

		} else {
				//
				// Verify that a single "+" or "-" sign exists as the
				// first character of the string.
				//
			var rRE = /^((\+|-){1}.+)$/;
	    	if (rRE.test(psValue)) {
					//
					// Strip the sign from the string.  Then, since
					// the remaining part of the string must meet the
					// requirements of an unsigned floating point
					// string, pass the string to the function
					// isFloatUnsigned.
					// 		
				var sValueStrippedOfSign = psValue.substr(1);
				return isFloatUnsigned(sValueStrippedOfSign, pbEmptyOK);
			
			} else {

				return false;
			
			}
		}
	}


	
		// This function receives a string and returns true if it is a
		// signed floating point decimal number within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		// For the definition of a signed floating point decimal
		// number, see comments associated with the function
		// isFloatSigned.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isFloatSignedInRange (psValue, pnMinValue, pnMaxValue,
											pbExclusive, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			// This verification must be done here at the start of the
			// function, rather than via the call below to function
			// isFloatSigned, since pbEmptyOK or
			// bNumberDefaultEmptyOK could allow an empty psValue
			// to be acceptable, which could cause incorrect operation of
			// the limit checking logic below.
			//													
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {
	
				return (bNumberDefaultEmptyOK == true);
			}

		} else {
				//
				// Verify that the argument to be validated is a valid
				// signed floating point number.  pbEmptyOK does not
				// need to be passed to isFloatSigned, since checking
				// for an empty psValue was performed above.
				//	
			if (isFloatSigned (psValue)) {
					//
					// For the limit check, the default is to include
					// the minimum and maximum limits, rather than to
					// exclude them.
					//
				if (isEmpty(pbExclusive) || pbExclusive != true) {
						//
					    // Check minimum (inclusively)
						//
				    if (!isEmpty(pnMinValue)) {
				        if (parseFloat(psValue) < pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (inclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
					
						if (parseFloat(psValue) > pnMaxValue) {
							return false;
						}
					}
	
				    return true;
					
				} else {
						//
					    // Check minimum (exclusively)
						//
				    if (!isEmpty(pnMinValue)) {
		
				        if (parseFloat(psValue) <= pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (exclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
		
						if (parseFloat(psValue) >= pnMaxValue) {
							return false;
						}
					}
					
				    return true;			
				
				}		
				
			} else {
			
				return false;
				
			}
		}
	}
	

		// This function receives a string and returns true if it is
		// either an unsigned floating point decimal number or a
		// signed floating point decimal number.
		// For the definitions of an unsigned floating point decimal
		// number and a signed floating point decimal number,
		// see comments associated with the functions isFloatUnsigned
		// and isFloatSigned.
		// 		
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isFloat (psValue, pbEmptyOK) {

		return (	isFloatUnsigned (psValue, pbEmptyOK) 	||
					isFloatSigned (psValue, pbEmptyOK)		);
	}
	


		// This function receives a string and returns true if it is
		// either an unsigned floating point decimal number or a
		// signed floating point decimal number and it is within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		// For the definitions of an unsigned floating point decimal
		// number and a signed floating point decimal number,
		// see comments associated with the functions isFloatUnsigned
		// and isFloatSigned.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isFloatInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK) {
											
		return (	isFloatUnsignedInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK)		||
					isFloatSignedInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK)		);					
	}

	
	
		//
		// This function receives a string and returns true if it is an
		// unsigned decimal integer.  An unsigned decimal integer is
		// defined as follows:
		// 		
		//	(a)  A decimal number (not octal or hexadecimal), consisting
		//		 of one or more digits, where each digit is 0-9.
		//		 Note that, for octal numbers (written with a leading
		//		 zero), this function returns true, since the octal number
		//		 format can be considered to be an unsigned decimal integer
		//		 with a single leading zero.
		//	(b)  No leading "+" or "-" sign.
		//	(c)  No decimal point.
		//	(d)  No leading, intervening, or trailing spaces.
		//
		// An integer can contain one or more leading zeros.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isIntegerUnsigned (psValue, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			//	
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {

				return (bNumberDefaultEmptyOK == true);
			}

		} else {

			var rRE = /^\d+$/;
	    	return rRE.test(psValue);
		}
	}



		//
		// This function receives a string and returns true if it is an
		// unsigned decimal integer within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		// For the definition of an unsigned decimal integer,
		// see comments associated with the function isIntegerUnsigned.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isIntegerUnsignedInRange (psValue, pnMinValue, pnMaxValue,
											pbExclusive, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			// This verification must be done here at the start of the
			// function, rather than via the call below to function
			// isIntegerUnsigned, since pbEmptyOK or
			// bNumberDefaultEmptyOK could allow an empty psValue
			// to be acceptable, which could cause incorrect operation of
			// the limit checking logic below.
			//													
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {
	
				return (bNumberDefaultEmptyOK == true);
			}

		} else {
				//
				// Verify that the argument to be validated is a valid
				// unsigned decimal integer.  pbEmptyOK does not
				// need to be passed to isIntegerUnsigned, since checking
				// for an empty psValue was performed above.
				//	
			if (isIntegerUnsigned (psValue)) {
					//
					// For the limit check, the default is to include
					// the minimum and maximum limits, rather than to
					// exclude them.
					//
				if (isEmpty(pbExclusive) || pbExclusive != true) {
						//
					    // Check minimum (inclusively)
						//
				    if (!isEmpty(pnMinValue)) {
				        if (parseInt(psValue, 10) < pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (inclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
					
						if (parseInt(psValue, 10) > pnMaxValue) {
							return false;
						}
					}
	
				    return true;
					
				} else {
						//
					    // Check minimum (exclusively)
						//
				    if (!isEmpty(pnMinValue)) {
		
				        if (parseInt(psValue, 10) <= pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (exclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
		
						if (parseInt(psValue, 10) >= pnMaxValue) {
							return false;
						}
					}
					
				    return true;			
				
				}		
				
			} else {
			
				return false;
				
			}
		}
	}
	
	

		// This function receives a string and returns true if it is a
		// signed decimal integer.  A signed decimal integer is defined
		// as follows:
		// 		
		//	(a)  Must contain a leading "+" or "-" sign, followed by (b).
		//	(b)  An unsigned decimal integer string, as defined in the
		//		 function isIntegerUnsigned.
		//	(c)  No spaces between the sign and the unsigned decimal
		//		 integer string.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isIntegerSigned (psValue, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			//	
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {

				return (bNumberDefaultEmptyOK == true);
			}

		} else {
				//
				// Verify that a single "+" or "-" sign exists as the
				// first character of the string.
				//
			var rRE = /^((\+|-){1}.+)$/;
	    	if (rRE.test(psValue)) {
					//
					// Strip the sign from the string.  Then, since
					// the remaining part of the string must meet the
					// requirements of an unsigned integer
					// string, pass the string to the function
					// isIntegerUnsigned.
					// 		
				var sValueStrippedOfSign = psValue.substr(1);
				return isIntegerUnsigned(sValueStrippedOfSign, pbEmptyOK);
			
			} else {

				return false;
			
			}
		}
	}




		// This function receives a string and returns true if it is a
		// signed decimal integer within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		// For the definition of a signed decimal
		// integer, see comments associated with the function
		// isIntegerSigned.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isIntegerSignedInRange (psValue, pnMinValue, pnMaxValue,
											pbExclusive, pbEmptyOK) {
			//
			// Verify that the argument to be validated is not empty.
			// This verification must be done here at the start of the
			// function, rather than via the call below to function
			// isIntegerSigned, since pbEmptyOK or
			// bNumberDefaultEmptyOK could allow an empty psValue
			// to be acceptable, which could cause incorrect operation of
			// the limit checking logic below.
			//													
		if (isEmpty(psValue)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psValue.
				// Otherwise, use the global bNumberDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {

				return (pbEmptyOK == true);

			} else {
	
				return (bNumberDefaultEmptyOK == true);
			}

		} else {
				//
				// Verify that the argument to be validated is a valid
				// signed integer.  pbEmptyOK does not
				// need to be passed to isIntegerSigned, since checking
				// for an empty psValue was performed above.
				//	
			if (isIntegerSigned (psValue)) {
					//
					// For the limit check, the default is to include
					// the minimum and maximum limits, rather than to
					// exclude them.
					//
				if (isEmpty(pbExclusive) || pbExclusive != true) {
						//
					    // Check minimum (inclusively)
						//
				    if (!isEmpty(pnMinValue)) {
				        if (parseInt(psValue, 10) < pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (inclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
					
						if (parseInt(psValue, 10) > pnMaxValue) {
							return false;
						}
					}
	
				    return true;
					
				} else {
						//
					    // Check minimum (exclusively)
						//
				    if (!isEmpty(pnMinValue)) {
		
				        if (parseInt(psValue, 10) <= pnMinValue) {
							return false;
						}
					}
						//
					    // Check maximum (exclusively)
						//
				    if (!isEmpty(pnMaxValue)) {
		
						if (parseInt(psValue, 10) >= pnMaxValue) {
							return false;
						}
					}
					
				    return true;			
				
				}		
				
			} else {
			
				return false;
				
			}
		}
	}
	

	

		// This function receives a string and returns true if it is
		// either an unsigned decimal integer or a
		// signed decimal integer.
		// For the definitions of an unsigned decimal
		// integer and a signed decimal integer,
		// see comments associated with the functions isIntegerUnsigned
		// and isIntegerSigned.
		// 		
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isInteger (psValue, pbEmptyOK) {

		return (	isIntegerUnsigned (psValue, pbEmptyOK) 	||
					isIntegerSigned (psValue, pbEmptyOK)		);
	}
	


		// This function receives a string and returns true if it is
		// either an unsigned decimal integer or a
		// signed decimal integer and it is within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		// For the definitions of an unsigned decimal
		// integer and a signed decimal integer,
		// see comments associated with the functions isIntegerUnsigned
		// and isIntegerSigned.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isIntegerInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK) {
											
		return (	isIntegerUnsignedInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK)		||
					isIntegerSignedInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK)		);					
	}

	

		//
		// This function receives a string and returns true if it is a
		// number in a specified radix.
		//
		// The default radix is 10 (decimal) (and, currently, only
		// decimal validation is available).
		// Octal (radix = 8) and hexadecimal (radix = 16) validation 
		// can be added in the future.
		//
		// For the string to be a number in radix 10, it must be
		// either a floating point decimal number or a decimal integer.
		// For the definitions of a floating point decimal
		// number and a decimal integer,
		// see comments associated with the functions isFloat
		// and isInteger.
		// 		
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the Boolean value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isNumber (psValue, pbEmptyOK, pnRadix) {
			//
			// The default radix is 10 (decimal) (and, currently, only
			// decimal validation is available).
			//
		if (isEmpty(pnRadix) || pnRadix == 10) {
	
			return (	isFloat (psValue, pbEmptyOK) 	||
						isInteger (psValue, pbEmptyOK)		);
		
		} else {

			return false;
		
		}
	}



		//
		// This function receives a string and returns true if it is a
		// number in a specified radix and it is within the limits
		// passed to the function as arguments.  The limiting arguments
		// (pnMinValue, pnMaxValue, and pbExclusive) are optional.
		//
		// The default radix is 10 (decimal) (and, currently, only
		// decimal validation is available).
		// Octal (radix = 8) and hexadecimal (radix = 16) validation 
		// can be added in the future.
		//
		// If the limit pnMinValue exists, this function will verify
		// that psValue is greater than or equal to pnMinValue.
		//
		// If the limit pnMaxValue exists, this function will verify
		// that psValue is less than or equal to pnMaxValue.
		//
		// By setting pbExclusive to true, this function will exclude
		// the limits from the allowable range.
		//
		// For the string to be a number in radix 10, it must be
		// either a floating point decimal number or a decimal integer.
		// For the definitions of a floating point decimal
		// number and a decimal integer,
		// see comments associated with the functions isFloat
		// and isInteger.
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// Note that this function requires that psValue be a string.  For
		// the following non-string values of psValue, this function returns
		// the following (assuming that bNumberDefaultEmptyOK = false and
		// pbEmptyOK = null):
		//
		// 		null									Returns false
		//		NaN (Not-a-Number)						Returns false
		//		true (logical value)					Returns false
		//		false (logical value)					Returns false
		//		Infinity (top-level property)			Returns false
		//
		// For other non-string values of psValue (including numbers
		// not treated as string literals) the value returned by this
		// function is indeterminate (due to number conversions made
		// internally by JavaScript).
		//
	function isNumberInRange (psValue, pnMinValue,
								pnMaxValue,	pbExclusive, pbEmptyOK, pnRadix) {
			//
			// The default radix is 10 (decimal) (and, currently, only
			// decimal validation is available).
			//
		if (isEmpty(pnRadix) || pnRadix == 10) {
											
			return (	isFloatInRange (psValue, pnMinValue,
									pnMaxValue,	pbExclusive, pbEmptyOK)		||
						isIntegerInRange (psValue, pnMinValue,
									pnMaxValue,	pbExclusive, pbEmptyOK)		);
		} else {
		
			return false;
			
		}
			
	}

	