/*
		Facility:
			SpatialObjects - Geographic Network Management
	        Copyright 2001 Celeritas Technologies, L.L.C.
			
	 	Title:
	   		ctValidateDate.js
			
		Abstract:
			This file contains various functions that provide
			validation of dates.  Most of these functions are intended
			to validate dates only when the dates are expressed as
			string literals.  See the comments associated with each
			individual function.
	      
	 	Author:
			Robert L. Marshall/Celeritas Technologies, LLC
	
	  	Creation Date:
	 		25-June-2001
	
		Modification History:
			<dd-mmm-yyyy>	<iii>
				<Description>
							
		See Also:
			ctValidateTest.html
			ctCalendarPopUp.js
	
		Dependencies:
			ctValidateCommon.js
			ctValidateNumber.js
*/

/*
	----------------------------------------------------------------------
	The test file for this file is ctValidateTest.html.  If any
	changes are made to the functions in this file, those changes must be
	tested by running the test file.
	----------------------------------------------------------------------
	
	pbEmptyOK:
	Some of the functions in this file provide the optional argument
	pbEmptyOK.  If a function provides argument pbEmptyOK, when a call to
	that function is made, if argument pbEmptyOK is not empty, pbEmptyOK
	(set to true or false) is used to set the function's return value
	when psValue is empty.  If pbEmptyOK is an empty argument, the
	global default variable	bNumberDefaultEmptyOK is used to set the
	function's return value.
	
	Global default variable bDateDefaultEmptyOK is initialized in
	file ctValidateCommon.js.
	
	For the meaning of "empty," see the function isEmpty in file
	ctValidateCommon.js.
	
*/

		//
		// This function receives a string (psDate) and returns true if it is a
		// date in the format specified by the optional format argument
		// (psDateFormat).  If psDateFormat is not supplied, a default
		// date format is used.
		// 
		// The allowable format characters are based on the format
		// characters used by Java's SimpleDateFormat class, and are a subset
		// of the SimpleDateFormat characters.
		//  	yyyy - Year (four-digit:  1000-9999)
		//		yy - Year 	(two-digit:  00-99)
		//      MMM - Month in year 	(One of the following (ignoring case):
		//									JAN, FEB, MAR, APR, MAY, JUN,
		//									JUL, AUG, SEP, OCT, NOV, DEC)
		//      MM - Month in year		(two-digit:  00-12)
		//      dd - Day in month 		(two-digit:  00-28, 29, 30, 31)
		//      HH - Hour in day 		(two-digit:  00-23)
		//      mm - Minute in hour 	(two-digit:  00-59)
		//      ss - Second in minute 	(two-digit:  00-59)
		//		SSS - Millisecond 		(three-digit:  000-999)
		//
		// Each of the above formatting codes can appear only once in
		// the format string.  All formatting codes are optional.
		//
		// For the delimiters that separate the date formatting codes,
		// any non-numeric, non-alphabetic characters can be used.  For
		// example, all of the following characters (including the space)
		// are valid as delimiters:
		//		/ .-_
		// 
		// The optional error message string (psErrorMessage) is appended
		// to the end of the context-specific error messages.
		//
		// If the optional flag pbDelimiterIgnore is true, the function
		// will not verify that the delimiters of psDate are the same
		// characters as the delimiters in psDateFormat.  This flag
		// provides the ability for this function to validate that all
		// date information and date field spacing is correct, without
		// forcing the user to re-enter the date simply because the user
		// typed an incorrect delimiter character while typing the rest
		// of the date correctly.  For an example of using pbDelimiterIgnore
		// in this way, see function ctCalendarShow in file ctCalendarPopUp.js.
		// pbDelimiterIgnore can be used in conjunction with pbDelimiterAdjust
		// (discussed below) to adjust automatically the delimiters, as
		// required.
		//
		// If the optional flag pbMessagesSuppress is true, all alert
		// messages are prevented from appearing, which allows this
		// function to be used for pure validation of dates (without
		// feedback to the user).
		//
		// For an explanation of the optional argument pbEmptyOK,
		// see the comments at the beginning of this file.
		//
		// If the optional flag pbDelimiterAdjust is true, each delimiter
		// in psDate will automatically be converted to agree with the
		// corresponding delimiter in psDateFormat.  However, note that if
		// pbDelimiterIgnore is not set to true and the delimiters in
		// psDate do not agree with those in psDateFormat, pbDelimiterAdjust
		// will have no effect, since this function will return false prior
		// to execution of the delimiter adjustment logic.  Similarly, note
		// that, if psDate is not a valid date (at least one of the
		// individual date elements, such as day, is not valid) this
		// function will return false prior to execution of the delimiter
		// adjustment logic.
		//
		// prFormObject is the reference to the object to which the
		// delimiter-adjusted date will be returned or to which focus
		// will be returned if psDate is not valid.  If pbDelimiterAdjust
		// or pbFocusReturn is set to true, prFormObject must be included
		// in the call to this function.
		//
		// If the optional flag pbFocusReturn is true, when psDate
		// is not valid (or the format provided in the function call is not
		// valid), focus is returned to the field specified by prFormObject.
		//
		// Portions of this function's code were adapted from functions in
		// ctDateValidate.js.
		//
	function isDate (psDate, psDateFormat, psErrorMessage,
									pbDelimiterIgnore, pbMessagesSuppress,
									pbEmptyOK, pbDelimiterAdjust,
									prFormObject, pbFocusReturn) {
			//
			// Set the default error message to an empty string.
			// If psErrorMessage exists, override the default.
			//
		var sErrorMessage = "";
		if (psErrorMessage != null) {
			sErrorMessage = "" + psErrorMessage;
		}
			//
			// Set the default value for bMessagesSuppress.
			// If pbMessagesSuppress is not empty, override the default.
			//
		var bMessagesSuppress = false;
		if (!isEmpty(pbMessagesSuppress)) {
			bMessagesSuppress = (pbMessagesSuppress == true);
		}
			//
			// Set the default value for rFormObject.
			// If prFormObject is not empty, override the default.
			//
		var rFormObject = null;
		if (!isEmpty(prFormObject)) {
			rFormObject = prFormObject;
		}
			//
			// Set the default value for bFocusReturn.
			// If pbFocusReturn is not empty, override the default.
			//
		var bFocusReturn = false;
		if (!isEmpty(pbFocusReturn)) {
			bFocusReturn = (pbFocusReturn == true);
		}
			//
			// Verify that the argument to be validated is not empty.
			//	
		if (isEmpty(psDate)) {
				//
				// If function argument pbEmptyOK is not empty, use it to
				// set the function's return value for the empty psDate.
				// Otherwise, use the global bDateDefaultEmptyOK to set
				// the function's return value.
				// 
			if (!isEmpty(pbEmptyOK)) {
			
				if ( (pbEmptyOK == true) != true &&
						(bMessagesSuppress == false) )	{
					alert("Date cannot be empty.\n" + sErrorMessage);
				}
				if ( (pbEmptyOK == true) != true ) {
					focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
				}
				return (pbEmptyOK == true);

			} else {
			
				if ( (bDateDefaultEmptyOK == true) != true &&
						(bMessagesSuppress == false) )	{
					alert("Date cannot be empty.\n" + sErrorMessage);
				}
				if ( (bDateDefaultEmptyOK == true) != true ) {
					focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
				}
				return (bDateDefaultEmptyOK == true);
			}

		} else {
			
			var sREFormat = "^";
			var sChar = "";
			var sMonthAbbrevs = "JANFEBMARAPRMAYJUNJULAUGSEPOCTNOVDEC";
			
			var nPosYearFourDigit = null;
			var nPosYearTwoDigit = null;
			var nPosMonthThreeChar = null;
			var nPosMonthTwoDigit = null;
			var nPosDayTwoDigit = null;
			var nPosHourTwoDigit = null;
			var nPosMinuteTwoDigit = null;
			var nPosSecondTwoDigit = null;
			var nPosMillisecondThreeDigit = null;
			var nMonthAbbrevIndex = null;

			var sYearFourDigit = null;
			var sYearTwoDigit = null;
			var sMonthThreeChar = null;
			var sMonthTwoDigit = null;
			var sDayTwoDigit = null;
			var sHourTwoDigit = null;
			var sMinuteTwoDigit = null;
			var sSecondTwoDigit = null;
			var sMillisecondThreeDigit = null;
				//
				// Set the default date format.  If psDateFormat exists,
				// override the default.
				//
			var sDateFormat = DEFAULT_DATE_TIME_FORMAT;
			if (psDateFormat != null) {
				sDateFormat = psDateFormat;
			}
				//
				// Set the default value for bDelimiterIgnore.
				// If pbDelimiterIgnore is not empty, override the default.
				//
			var bDelimiterIgnore = false;
			if (!isEmpty(pbDelimiterIgnore)) {
				bDelimiterIgnore = (pbDelimiterIgnore == true);
			}
				//
				// Set the default value for bDelimiterAdjust.
				// If pbDelimiterAdjust is not empty, override the default.
				//
			var bDelimiterAdjust = false;
			if (!isEmpty(pbDelimiterAdjust)) {
				bDelimiterAdjust = (pbDelimiterAdjust == true);
			}
				//
				// Step through each character of sDateFormat,
				// building a reqular expression so that it matches 
				// sDateFormat.  Simultaneouly capture the character
				// positions of the formatting sequences in sDateFormat.
				//
			for (var nPosition = 0; nPosition < sDateFormat.length;
					nPosition++) {

				sChar = sDateFormat.charAt(nPosition);

				if (sChar == "y") {

					if (sDateFormat.substr(nPosition, 4) == "yyyy") {

						sREFormat += "\\d{4}";
						nPosYearFourDigit = nPosition;
						nPosition += 3;

					} else if (sDateFormat.substr(nPosition, 2) == "yy") {

						sREFormat += "\\d{2}";
						nPosYearTwoDigit = nPosition;
						nPosition += 1;										

					} else {
							//
							// Since the year format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {
							alert("Internal year format error.  " +
									"Contact support."); 
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}

				} else if (sChar == "M") {

					if (sDateFormat.substr(nPosition, 3) == "MMM") {

						sREFormat += "[A-Za-z]{3}";
						nPosMonthThreeChar = nPosition;
						nPosition += 2;

					} else if (sDateFormat.substr(nPosition, 2) == "MM") {

						sREFormat += "\\d{2}";
						nPosMonthTwoDigit = nPosition;
						nPosition += 1;										

					} else {
							//
							// Since the month format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {
							alert("Internal month format error.  " +
									"Contact support."); 
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}				

				} else if (sChar == "d") {

					if (sDateFormat.substr(nPosition, 2) == "dd") {

						sREFormat += "\\d{2}";
						nPosDayTwoDigit = nPosition;
						nPosition += 1;

					} else {
							//
							// Since the day format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {							
							alert("Internal day format error.  " +
									"Contact support.");
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject); 
						return false;
					}

				} else if (sChar == "H") {

					if (sDateFormat.substr(nPosition, 2) == "HH") {

						sREFormat += "\\d{2}";
						nPosHourTwoDigit = nPosition;
						nPosition += 1;

					} else {
							//
							// Since the hour format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {							
							alert("Internal hour format error.  " +
									"Contact support."); 
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}

				} else if (sChar == "m") {

					if (sDateFormat.substr(nPosition, 2) == "mm") {

						sREFormat += "\\d{2}";
						nPosMinuteTwoDigit = nPosition;
						nPosition += 1;

					} else {
							//
							// Since the minute format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {
							alert("Internal minute format error.  " +
									"Contact support."); 
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}

				} else if (sChar == "s") {

					if (sDateFormat.substr(nPosition, 2) == "ss") {

						sREFormat += "\\d{2}";
						nPosSecondTwoDigit = nPosition;
						nPosition += 1;

					} else {
							//
							// Since the second format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {
							alert("Internal second format error.  " +
									"Contact support."); 
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}

				} else if (sChar == "S") {

					if (sDateFormat.substr(nPosition, 3) == "SSS") {

						sREFormat += "\\d{3}";
						nPosMillisecondThreeDigit = nPosition;
						nPosition += 2;

					} else {
							//
							// Since the millisecond format is not one of the
							// allowable formats, display an error
							// message, and return false.
							//
						if (bMessagesSuppress == false) {							
							alert("Internal millisecond format error.  " +
									"Contact support."); 
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}

				} else {
						//
						// If the flag is set not to ensure that
						// the delimiters of psDate are the same
						// characters as the delimiters in sDateFormat,
						// set the regular expression mask to allow any
						// non-alphanumeric character for the delimiter
						// in psDate.  Otherwise, set the regular
						// expression mask to require that each delimiter
						// be the same as its corresponding delimiter in
						// sDateFormat.
						//
					if (bDelimiterIgnore == true) {
						sREFormat += "[^A-Za-z0-9]";					
					} else {
						sREFormat += "\\" + sChar;
					}
				}	

			}
		
			sREFormat += "$";		
				//
				// Using the regular expression that was just generated,
				// test whether psDate matches the regular expression.
				// If it does not match, return false.  If it matches,
				// validate the individual date field values. 
				//
			var rREFormat = new RegExp(sREFormat);
			if (!rREFormat.test(psDate)) {

				if (bMessagesSuppress == false) {

					alert("Please format date as:  \"" + sDateFormat + "\"");
				}
				focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
				return false;
				//
				// Since, psDate has the correct date format,
				// validate the values of the individual date elements
				// of psDate.
				//
			} else {
					//
					// If the date format indicates that the year (four-digit)
					// field is to exist, validate its value.
					//
				if (nPosYearFourDigit != null) {
					sYearFourDigit = psDate.substr(nPosYearFourDigit, 4);
					if (!isIntegerUnsignedInRange(sYearFourDigit, 1000, 9999,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Year \"" + sYearFourDigit + "\" must be " +
									"four digits (0-9) and in the range " +
									"1000-9999.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}				
					//
					// If the date format indicates that the year (two-digit)
					// field is to exist, validate its value.
					//
				if (nPosYearTwoDigit != null) {
					sYearTwoDigit = psDate.substr(nPosYearTwoDigit, 2);
					if (!isIntegerUnsignedInRange(sYearTwoDigit, 0, 99,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Year \"" + sYearTwoDigit + "\" must be " +
									"two digits (0-9) and in the range " +
									"00-99.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}	
					//
					// If the date format indicates that the month
					// (three-character) field is to exist, validate its
					// value.  The abbreviation is valid regardless of
					// whether its individual letters are upper or lower
					// case (so functions using this function may need
					// to rebuild the abbreviation to force the letters
					// to the correct case--See the date string rebuilding
					// comments at the beginning of this function).
					//
				if (nPosMonthThreeChar != null) {
					sMonthThreeChar = psDate.substr(nPosMonthThreeChar, 3);
					nMonthAbbrevIndex = sMonthAbbrevs.indexOf(
							sMonthThreeChar.toUpperCase());
					if (nMonthAbbrevIndex < 0) {
						if (bMessagesSuppress == false) {
							alert("Month \"" + sMonthThreeChar +
									"\" must be three abbreviation " +
									"characters.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;
					}
				}	
					//
					// If the date format indicates that the month (two-digit)
					// field is to exist, validate its value.
					//
				if (nPosMonthTwoDigit != null) {
					sMonthTwoDigit = psDate.substr(nPosMonthTwoDigit, 2);
					if (!isIntegerUnsignedInRange(sMonthTwoDigit, 1, 12,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Month \"" + sMonthTwoDigit + "\" must be " +
									"two digits (0-9) and in the range " +
									"01-12.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}	
					//
					// If the date format indicates that the day (two-digit)
					// field is to exist, validate its value.
					// Validation of the day uses valid month and year
					// values obtained from the code above.
					//
					// If month information is not part of the
					// date format, or if the month is February and the
					// year is not part of the date format, the day is just
					// validated that it is no greater than the default
					// nMaxDay (=31).
					//
				if (nPosDayTwoDigit != null) {
					sDayTwoDigit = psDate.substr(nPosDayTwoDigit, 2);
					nMaxDay = 31;
					var nYearForDayCalculation = null;
					var nMonthForDayCalculation = null;
						//
						// Obtain the year associated with this day.
						//
					if (sYearFourDigit != null) {
						nYearForDayCalculation = parseInt(sYearFourDigit);
					} else if (sYearTwoDigit != null) {
						nYearForDayCalculation = parseInt(
								adjustYearTwoDigitToFourDigit (sYearTwoDigit));
					}
						//
						// Obtain the month associated with this day.
						//
					if (sMonthThreeChar != null) {
							//
							// Convert the month abbreviation into a
							// month number.
							//
						nMonthForDayCalculation = nMonthAbbrevIndex/3 + 1;
					} else if (sMonthTwoDigit != null) {
						nMonthForDayCalculation = parseInt(sMonthTwoDigit);
					}
						//
						// Use valid month and year values obtained from
						// the code above to determine the maximum value
						// for the day.
						// If month information is not part of the
						// date format, or if the month is February and the
						// year is not part of the date format, the day is just
						// validated that it is no greater than the default
						// nMaxDay from above.
						//
						//
						// April, June, September, and November
						// always have 30 days.
						//
					if (nMonthForDayCalculation == 4 ||
							nMonthForDayCalculation == 6 ||
							nMonthForDayCalculation == 9 ||
							nMonthForDayCalculation == 11) {
						nMaxDay = 30;
							//
							// For the number of days in February,
							// determine whether the year is a leap year.
							//
					} else if (nMonthForDayCalculation == 2 &&
							nYearForDayCalculation != null)	{

						if ( (nYearForDayCalculation % 4 > 0) ||
									(nYearForDayCalculation % 100 == 0 &&
										nYearForDayCalculation % 400 > 0)	) {
								// Non-leap year.
							nMaxDay = 28;
						} else {
								// Leap year.
							nMaxDay = 29;
						}
					}
					
					if (!isIntegerUnsignedInRange(sDayTwoDigit, 1, nMaxDay,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Day \"" + sDayTwoDigit + "\" must be " +
									"two digits (0-9) and in the range valid " +
									"for the other date parameters " +
									"being provided.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}	

					//
					// If the hour format indicates that the hour (two-digit)
					// field is to exist, validate its value.
					//
				if (nPosHourTwoDigit != null) {
					sHourTwoDigit = psDate.substr(nPosHourTwoDigit, 2);
					if (!isIntegerUnsignedInRange(sHourTwoDigit, 0, 23,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Hour \"" + sHourTwoDigit + "\" must be " +
									"two digits (0-9) and in the range " +
									"00-23.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}		
					//
					// If the minute format indicates that the minute
					// (two-digit) field is to exist, validate its value.
					//
				if (nPosMinuteTwoDigit != null) {
					sMinuteTwoDigit = psDate.substr(nPosMinuteTwoDigit, 2);
					if (!isIntegerUnsignedInRange(sMinuteTwoDigit, 0, 59,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Minute \"" + sMinuteTwoDigit +
									"\" must be " +
									"two digits (0-9) and in the range " +
									"00-59.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}	
					//
					// If the second format indicates that the second
					// (two-digit) field is to exist, validate its value.
					//
				if (nPosSecondTwoDigit != null) {
					sSecondTwoDigit = psDate.substr(nPosSecondTwoDigit, 2);
					if (!isIntegerUnsignedInRange(sSecondTwoDigit, 0, 59,
												false, false)) {
						if (bMessagesSuppress == false) {
							alert("Second \"" + sSecondTwoDigit +
									"\" must be " +
									"two digits (0-9) and in the range " +
									"00-59.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}	
					//
					// If the millisecond format indicates that the millisecond
					// (three-digit) field is to exist, validate its value.
					//
				if (nPosMillisecondThreeDigit != null) {
					sMillisecondThreeDigit = psDate.substr(
							nPosMillisecondThreeDigit, 3);
					if (!isIntegerUnsignedInRange(sMillisecondThreeDigit,
											0, 999,	false, false)) {
						if (bMessagesSuppress == false) {
							alert("Millisecond \"" + sMillisecondThreeDigit +
									"\" must be three digits (0-9) and in " +
									"the range 000-999.\n" + sErrorMessage);
						}
						focusOnFormObjectIfRequired(bFocusReturn, rFormObject);
						return false;							
					}
				}	
					//
					// If none of the conditions in the code above caused
					// a return false, psDate consists of valid date
					// elements, so continue below with the available return
					// options and then return true from this function.
					//
					// Since psDate was validated above to be
					// (character-for-character) in the same
					// format as sDateFormat (except possibly for
					// the delimiters, if bDelimiterIgnore caused the
					// delimiters to be ignored), if automatic delimiter
					// adjustment is to be performed, step through each
					// character of psDate, adjusting the delimiters to
					// conform to the corresponding delimiters of
					// psDateFormat.  Then update the form object with
					// the delimiter-adjusted date.
					//
				var sDateForReturn = psDate;
				if ( (pbDelimiterAdjust == true) && (rFormObject != null) ) {

					var rREDelimiterFormat = new RegExp("[^A-Za-z0-9]");				
					for (var nPositionForDelimiterAdjust = 0;
							nPositionForDelimiterAdjust < sDateForReturn.length;
							nPositionForDelimiterAdjust++) {

						sCharFromDate =	sDateForReturn.charAt(
								nPositionForDelimiterAdjust);
							//
							// If the character is a delimiter, overwrite it
							// with the corresponding delimiter from
							// pDateFormat.
							//
						if (rREDelimiterFormat.test(sCharFromDate)) {

							sDateForReturnBegin =
									sDateForReturn.substring(
											0, nPositionForDelimiterAdjust);
							sDateForReturnEnd =
									sDateForReturn.substr(
											nPositionForDelimiterAdjust +1);
							sDateForReturn =
										sDateForReturnBegin +
										sDateFormat.charAt(
												nPositionForDelimiterAdjust) +
										sDateForReturnEnd;
						}
					}

					rFormObject.value = sDateForReturn;
				}
					
				return true;		
			}
		}
		
	}
	
	
		//		
		// This function provides support to the function isDate.
		// It provides the ability to give focus to a particular form
		// object.
		//
	function focusOnFormObjectIfRequired(pbFocusRequired, prFormObject) {

		if ( (pbFocusRequired == true) && (prFormObject != null) ) {
			prFormObject.select();
			prFormObject.focus();
		}
	}
		
	
	
	
		//
		// This function receives a string ("yy") representing a year (in
		// two-digit form--that is, not including the century digits "cc"
		// of the full form "ccyy") and returns the four-digit form ("ccyy")
		// of the year.
		// The decision for which century digits to provide for the prefix
		// "cc" is based on the criteria used by the Java class
		// SimleDateFormat; that is, the function sets the year to
		// be within 80 years before and 20 years after the time at
		// which the function is executed.
		//
		// Note that this function requires that psYearTwoDigit is a string.
		// Also, if psYearTwoDigit is a single digit, this function will
		// provide a single leading zero to pad the year.
		// If psYearTwoDigit is not in the range 00-99, or if it is more
		// than two characters, this function just returns psYearTwoDigit.
		// 
	function adjustYearTwoDigitToFourDigit (psYearTwoDigit) {
			//
			// If psYearTwoDigit is not an unsigned integer in
			// the required range, or if psYearTwoDigit is more than
			// two characters, just return psYearTwoDigit.
			//
		var rREMoreThanTwoChar = /^.{3,}$/;
		if (!isIntegerUnsignedInRange(psYearTwoDigit, 0, 99, false, false) ||
				rREMoreThanTwoChar.test(psYearTwoDigit)) {
			return psYearTwoDigit;
		} else {
				
			var sYearTwoDigit = psYearTwoDigit;
				//
				// If sYearTwoDigit is only a single decimal digit,
				// pad it with a leading zero.
				//
			var rRE = /^\d{1}$/;
			if (rRE.test(sYearTwoDigit)) {
				sYearTwoDigit = "0" + sYearTwoDigit;
			}
				//
				// Get the current date/time, and use its century digits
				// to develop the year that is being adjusted to four digits.
				//
			var dNow = new Date();
			var sYearFourDigit =
					"" + dNow.getFullYear().toString().substr(0, 2) +
					sYearTwoDigit;
				//
				// Create a date object and set its year equal to the
				// year being adjusted to four digits.
				//
			var dDateFromTwoDigits = new Date(dNow.getTime());
			dDateFromTwoDigits.setFullYear(sYearFourDigit);
				//
				// If dDateFromTwoDigits is before or equal to dNow and
				// the difference in time between them is greater than or
				// equal to 80 years, add one century to the year being
				// returned.
				//
			var nYearToReturn = parseInt(sYearFourDigit);
			
			if (dDateFromTwoDigits <= dNow) {

				var dEightyYearsAgo = new Date(dNow.getTime());
				dEightyYearsAgo.setFullYear(dNow.getFullYear() - 80);

				if (dDateFromTwoDigits <= dEightyYearsAgo) {
					nYearToReturn += 100;
				}
			}
				//
				// If dDateFromTwoDigits is after dNow and
				// the difference in time between them is greater than
				// 20 years, subtract one century from the year being
				// returned.
				//
			if (dDateFromTwoDigits > dNow) {
				var dTwentyYearsAhead = new Date(dNow.getTime());
				dTwentyYearsAhead.setFullYear(dNow.getFullYear() + 20);

				if (dDateFromTwoDigits > dTwentyYearsAhead) {
					nYearToReturn -= 100;
				}
			}
				// Since nYearToReturn is always greater than
				// 1900 (since its two century digits are no more than
				// one century prior to the time this function is invoked),
				// nYearToReturn always consists of four digits;
				// Therefore, there is no need to pad with leading zeros
				// the year being returned.
				//
			return nYearToReturn.toString();
		}
		
	}
	