/*
    Facility:
        Celeritas Technologies (ct) 
		Copyright 2001 Celeritas Technologies, LLC

    Title:
        ctCalendarPopUp.js

    Abstract:
        This module contains routines to build and manipulate a calendar.  Users
		can utilize the calendar to select a date value for a date input field.
		The calendar is displayed in a pop-up browser window.
       
    Requires:
        ctPopUpDialog.js
		ctValidateDate.js
		ctValidateNumber.js
		ctValidateCommon.js

    Author:
        Scott A. Evans (sae)
        
    Creation Date:
        29-Mar-2001
        
    Modification History:
		23-Jul-2001		(rlm)
            Added conditional logic to ctCalendarShow(prWinObj) to set
			the calendar's initial date to today's date if the value
			obtained from the opener window's return field is not a valid date.
			Changed function name ctCalendareDateReplace to
			ctCalendarDateReplace.
			Added logic to support two-digit year and three-character month.
			Added an error message for the case when the year is outside
			the range able to be displayed by the calendar.
		27-Jul-2001		(rlm)
			Removed from ctCalendarChangeYear(prYearField, prWinObj) the
			logic that forced the month to change to January (or the current
			month) when the year was changed to the non-current year (or
			to the current year).  Now, when the year is changed, the month
			is not modified.
		22-Aug-2001		(dws)
			Extensive modifications to allow different types of calendars to be
			displayed.  Allowable types are daily (default), weekly, monthly,
			quarterly, and yearly.
		26-Feb-2002		(dws)
			Modified function ctCalendarDateReplace to use a Regular Expression to
			create the returned date string.  This was done to allow the calendar
			to work correctly on Netscape6.  For some reason, the Javascript 
			string function 'split' did not work the same on Netscape and IE.

	Notes:
		The allowable format characters are based on the format
		characters used by Java's SimpleDateFormat class, and are a subset
		of the SimpleDateFormat characters.  (For the range of allowable
		years, see the constants CALENDAR_BASE_YEAR and CALENDAR_MAX_YEAR.)
 				yyyy - Year (four-digit)
				yy - Year 	(two-digit)
     			MMM - Month in year 	(One of the following (ignoring case):
											JAN, FEB, MAR, APR, MAY, JUN,
											JUL, AUG, SEP, OCT, NOV, DEC)
     			MM - Month in year		(two-digit:  00-12)
		        dd - Day in month 		(two-digit:  00-28, 29, 30, 31)
		
		Each of the above formatting codes can appear only once in
		the format string.  In general, the format string should contain
		a formatting code for each of the following date elements:  day,
		month, and year.  If a formatting code does not appear for day,
		month, and year, the value for the missing date element(s) is
		assumed to be that corresponding to the current day.
	
		For the delimiters that separate the date formatting codes,
		any non-numeric, non-alphabetic characters can be used.  For
		example, all of the following characters (including the space)
		are valid as delimiters:
				/ .-_
		In the date returned by the calendar, the delimiters are converted
		to those used in the format string, regardless of what delimters
		appeared in the date string received by the calendar.
			
*/
	//
	// Non-Leap year Month days.
	//
var nDaysOfMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
	//
	// Leap year Month days.
	//
var nLeapDaysOfMonth = [31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];

var sMonthOfYear = ["January","February","March","April","May","June","July",
		"August","September","October","November","December"];
		
var	sMonthAbbrevs = "JANFEBMARAPRMAYJUNJULAUGSEPOCTNOVDEC";
var sMonthThreeChar = null;
var nMonthAbbrevIndex = null;
	//
	// CALENDAR_BASE_YEAR and CALENDAR_MAX_YEAR define the range of years
	// that will be displayed in the year selection box.
	// BASE_YEAR_BREAKPOINT is the two-digit year at which the century
	// representation changes from the 1900s to the 2000s and is used to
	// convert two-digit years to four-digit years.  Therefore, the
	// following relationship must be maintained:
	// 			CALENDAR_MAX_YEAR - CALENDAR_BASE_YEAR <= 99
	// For usage of BASE_YEAR_BREAKPOINT, see the function ctParseDate.
	//
var CALENDAR_BASE_YEAR = 1990;
var CALENDAR_MAX_YEAR = 2030;
var BASE_YEAR_BREAKPOINT = CALENDAR_BASE_YEAR - 1900;
var MILLIS_PER_DAY = 86400000;

var rCurrDate = new Date();
var nCurrDay = rCurrDate.getDate();
var nCurrMonth = rCurrDate.getMonth();
var nCurrYear = rCurrDate.getFullYear();

var nSelectedDay = nCurrDay;
var nSelectedMonth = nCurrMonth;
var nSelectedYear = nCurrYear;
var nStartingDay = nCurrDay;
var nStartingMonth = nCurrMonth;
var nStartingYear = nCurrYear;
	//
	// Variables used for displaying different types of calendars.
	//
var nDateRangeStart = 0;
var nDateRangeEnd = 0;
var nYearlyDisplayIncrement = 0;
var sCalendarType = "daily";
	//
	// bNewCalendar indicates that a popup calendar is being requested.
	//
var bNewCalendar = true;
	//
	// bErrorMessageDisplayed indicates that a previous attempt to display
	// a popup calendar resulted in an error message (rather than the
	// calendar) being displayed in the calendar window.
	//
var bErrorMessageDisplayed = false;

/*
 * Parses a date string to separate it into year, month and day parts.
 *
 * Arguments:
 * 		sDateFormat	- The date format to use for parsing the date string.
 * 		sDateString	- The date string to be parsed.
 *
 * Returns:
 *		An Array of date part values (in year, month, day order).
 */
function ctParseDate(sDateFormat, sDateString) {
	anDateParts = new Array(3);
		//
		// parse the Year
		//
	var sYear = "" + nCurrYear;
	nPos = sDateFormat.indexOf("yyyy");
	if ( nPos < 0 ) {
		nPos = sDateFormat.indexOf("yy");
		if ( nPos >= 0 ) {
			sYear = sDateString.substring(nPos, nPos + 2);
				//
				// Add the century digits to the beginning of the two-digit
				// year to convert it to a four-digit year.
				// BASE_YEAR_BREAKPOINT, the "rollover" breakpoint (from the
				// 1900s to the 2000s), determines whether "19" or "20" is
				// added to represent the century.
				//
			if ( parseInt(sYear, 10) < BASE_YEAR_BREAKPOINT ) {
				sYear = "20" + sYear;
			} else {
				sYear = "19" + sYear;
			}
		}

	} else {
		sYear = sDateString.substring(nPos, nPos + 4);
	}
	anDateParts[0] = parseInt(sYear, 10);
		//
		// parse the Month
		//
		// For use in this parse function, sMonth is initialized to a number
		// that is one greater than the current month number, since, upon
		// return from this parse function, nSelectedMonth is set to one
		// less than sMonth.  However, generally, this initialized value
		// will be overwritten by the code that follows.
		//
	var sMonth = "" + (nCurrMonth + 1);
	nPos = sDateFormat.indexOf("MMM");
	if ( nPos >= 0 ) {

	    sMonthThreeChar = sDateString.substring(nPos, nPos + 3);
		nMonthAbbrevIndex = sMonthAbbrevs.indexOf(
				sMonthThreeChar.toUpperCase());
			//
			// Convert the month abbreviation into a month number string.
			//
		sMonth = "" + (nMonthAbbrevIndex/3 + 1);

	} else {
		nPos = sDateFormat.indexOf("MM");
		if ( nPos >= 0 ) {
		    sMonth = sDateString.substring(nPos, nPos + 2);
		}
	}
	anDateParts[1] = parseInt(sMonth, 10);
		//
		// parse the Day
		//
	var sDay = "" + nCurrDay;
	nPos = sDateFormat.indexOf("dd");
	if ( nPos >= 0 ) {
	    sDay = sDateString.substring(nPos, nPos + 2);
	}
	anDateParts[2] = parseInt(sDay, 10);

	return anDateParts;
	
}

/*
 * Returns the number of days in a given month.
 *
 * Arguments:
 * 		pnMonthNumber	- The month of the year (zero based, i.e. 0..11).
 * 		pnYear			- The year currently selected on the calendar.
 *
 * Returns:
 *		The number of days in the specified month.
 */
function ctCalendarGetDaysOfMonth(pnMonthNumber, pnYear) {
		//
		// Compensate for leap year when determining the number of days
		// in the month.
		//
	if ((pnYear % 4) == 0) {
		if ((pnYear % 100) == 0 && (pnYear % 400) != 0) {
			return nDaysOfMonth[pnMonthNumber];
		}
		return nLeapDaysOfMonth[pnMonthNumber];
	} else
		return nDaysOfMonth[pnMonthNumber];
}

/*
 * Determines the color to use based on the day of the month.  One
 * color is used for the current day, another color is used for the
 * selected day, and another color is used for all other days.
 *
 * Arguments:
 * 		pnYear			- The year currently selected on the calendar.
 * 		pnMonthOfYear	- The month of the year (zero based, i.e. 0..11).
 *		pnDayOfMonth	- A day within the given month.
 *
 * Returns:
 *		The hex RGB color code to use for the given day.
 */
function ctCalendarGetDayColor(pnYear, pnMonthOfYear, pnDayOfMonth) {
	var rCurrDate = new Date();
	var nCurrDay = rCurrDate.getDate();
	var nCurrMonth = rCurrDate.getMonth();
		//
		// Set the current day values to allow current date colors to be displayed
		// for monthly, quarterly, and yearly calendar types.
		//
	if ( sCalendarType == 'monthly' || sCalendarType == 'quarterly' ) {
		nCurrDay = 1;
	} else if ( sCalendarType == 'yearly' ) {
		nCurrDay = 1;
		nCurrMonth = 0;
	}
		//
		// Create a date object for testing against a date range.
		//
	var sPassedDate = "" + (parseInt(pnMonthOfYear)+1) + "/" + pnDayOfMonth + "/" + pnYear;
	var dPassedDate = new Date(sPassedDate);
	
	if ((pnYear == nCurrYear ) &&
	    ( pnMonthOfYear == nCurrMonth ) && 
		( pnDayOfMonth == nCurrDay )) {
			//
			// If the date given is the current date, mark its
			// color differently so that it stands out.
			//
		return 'CCCCFF';
	} else if ((pnYear == nStartingYear ) &&
	    	   ( pnMonthOfYear == nStartingMonth ) && 
			   ( pnDayOfMonth == nStartingDay )) {
			//
			// Mark the currently selected date with a special color
			// to indicate what date we started with.
			//
		return 'CCFFCC';
	} else if ( ( nDateRangeStart <= dPassedDate.getTime() ) &&
			   ( dPassedDate.getTime() <= nDateRangeEnd ) ) {
			//
			// Mark dates falling within the date range with a special color
			// to indicate the selected date range.  This block is used
			// mainly with weekly and quarterly calender types.
			//
		return 'CCFFCC';
	} else {
			//
			// Nothing special about this date, just return the
			// calendar's standard background color.
			//
		return 'FFFFFF';
	}	
}

/*
 * Formats a portion of a date string.  For example this routine might be passed
 * a date string value of "MM/dd/2001", a match string value of "MM" and a date
 * part value of "12" which would result in a new date string of "12/dd/2001".
 *
 * Arguments:
 * 		psString	- The date string which may contain part format characters and
 *					  part previously formatted date values.
 * 		psMatchStr	- A token string used to match to a portion of the date format. 
 *		psDatePart	- A string representing a portion of a date (e.g. a month).
 *		bPad		- True if the date part is to be zero padded.
 *
 * Returns:
 *		A string representing a portion of a date
 */
function ctCalendarDateReplace(psString, psMatchStr, psDatePart, bPad) {

    if ( psString == null ) {
        return "";
    }
		//
    	// Replace the instance of the match string with the replacement 
		// string, padded to the specified number of digits.
		//
    if (psString.indexOf(psMatchStr) != -1) {

		var sReplaceStr = psDatePart;
    	if ( psMatchStr != "MMM" && bPad && parseInt(psDatePart, 10) < 10 ) {
			sReplaceStr = "0" + psDatePart;
		}
			//
        	// Create a regular expression from the match string, and use the
			// expression to replace the appropriate piece of the string format.
			//
		var re = new RegExp(psMatchStr);
        rResult = psString.replace(re, sReplaceStr);
			//
        	// Return the result with the replacement in place
			// of the match string.
			//
        return rResult;
    }
    else {
        return psString;
    }	
}

/*
 * Returns a date string, formatted to comply with a given date format pattern.
 *
 * Arguments:
 * 		psDateFormat	- The date format pattern.
 * 		pnDay			- The day of the month (number)
 *		pnMonth			- The month of the year (number)
 *		pnYear			- The year (four-digit year number)
 *
 * Returns:
 *		A a formatted date string.
 */
function ctCalendarFormatDate(psDateFormat, pnDay, pnMonth, pnYear) {
	var sDate = psDateFormat;
	var nPos = -1;
		//
		// Default the date format if no format was given.
		//
	if ( sDate == null ) {
			//
			// DEFAULT_DATE_FORMAT is defined in ctValidateCommon.js.
			// To provide correct default date format operation of the
			// dateInput custom tag, this default date format must be
			// consistent with the default date format in the custom tag
			// Java file DateInputTag.java.
			//
		sDate = DEFAULT_DATE_FORMAT;
	}
		//
		// Format the Day portion of the date.
		//
    if (sDate.indexOf("dd") != -1) {
        sDate = ctCalendarDateReplace(sDate, "dd", ("" + pnDay), true);
    }
		//
		// Format the month portion.
		//
    if ( sDate.indexOf("MMM") != -1 ) {

        sDate = ctCalendarDateReplace(sDate, "MMM",
				sMonthOfYear[pnMonth-1].substring(0,3), false);

    } else if (sDate.indexOf("MM") != -1) {

        sDate = ctCalendarDateReplace(sDate, "MM", ("" + pnMonth), true);
    }
		//
		// Format the year portion.
		//
    if ( sDate.indexOf("yyyy") != -1 ) {
        sDate = ctCalendarDateReplace(sDate, "yyyy", ("" + pnYear), false);
    } else if ( sDate.indexOf("yy") != -1 ) {
        var sYear = "" + pnYear;
        sYear = sYear.substring(2,4);
        sDate = ctCalendarDateReplace(sDate, "yy", sYear, false);
    }
	
	return sDate;
}

/*
 * Updates the contents of the calendar with a new date.
 *
 * Arguments:
 *		pnMonth		- The new month.
 *		pnYear		- The new year
 * 		prWinObj	- The window object which contains the calendar information.
 *
 * Returns:
 *		Void
 */
function ctCalendarUpdateCalendar(pnMonth, pnYear, prWinObj) {
	rCurrDate.setMonth(pnMonth);
	rCurrDate.setFullYear(pnYear);

	ctCalendarShow(prWinObj);
}

/*
 * Updates the month of the calendar and causes the calendar information
 * to be regenerated based on this new month.
 *
 * Arguments:
 *		prMonthField	- The form field containing the new month.
 * 		prWinObj		- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarChangeMonth(prMonthField, prWinObj) {
	nSelectedMonth = prMonthField.selectedIndex;
	ctCalendarUpdateCalendar(nSelectedMonth, nSelectedYear, prWinObj);
}

/*
 * Updates the year of the calendar and causes the calendar information
 * to be regenerated based on this new year.
 *
 * Arguments:
 *		prYearField	- The form field containing the new year.
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarChangeYear(prYearField, prWinObj) {
	nSelectedYear = prYearField.selectedIndex + CALENDAR_BASE_YEAR;
	ctCalendarUpdateCalendar(nSelectedMonth, nSelectedYear, prWinObj);
}

/*
 * Set the designated date input field with the formatted date selected from
 * the calendar.  The displayed date will depend on the type of Calendar being
 * displayed.  In general, the first day of the requested time period will be
 * the date displayed.  For example, if the type is 'weekly', then any day selected
 * within a given week will return the first day of that week.
 *
 * Arguments:
 *		pnDay		- A day selected from the current calendar.
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarSelectDay(pnDay, prWinObj) {

	if ( sCalendarType == 'weekly' ) {
			//
			// A date range will created using the selected date.  The returned value
			// will be the first day of the week containing the selected day.
			//
		sStartingDate = "" + (parseInt(nSelectedMonth)+1) + "/" + pnDay + "/" + nSelectedYear;
		ctCalendarSetDateRange();
		var dOutDate = new Date(nDateRangeStart);
		prWinObj.opener.__returnField.value = ctCalendarFormatDate(
			prWinObj.opener.__dateFormat, dOutDate.getDate(), parseInt(dOutDate.getMonth())+1,
			dOutDate.getFullYear() );
	} else if ( sCalendarType == 'monthly' ) {
			//
			// The "day" is actually the selected month value.  The returned value will
			// be the first day of the selected month.
			//
		nSelectedMonth = pnDay;
		prWinObj.opener.__returnField.value = ctCalendarFormatDate(
			prWinObj.opener.__dateFormat, 1, nSelectedMonth,
			nSelectedYear);
	} else if ( sCalendarType == 'quarterly' ) {
			//
			// The "day" is actually the selected month value.  A date range will created
			// using a date of the first day of the selected month.  The returned value
			// will be the first day of the quarter containing the selected month.
			//
		nSelectedMonth = pnDay;
		sStartingDate = "" + nSelectedMonth + "/01/" + nSelectedYear;
		ctCalendarSetDateRange();
		var dOutDate = new Date(nDateRangeStart);
		prWinObj.opener.__returnField.value = ctCalendarFormatDate(
			prWinObj.opener.__dateFormat, dOutDate.getDate(), parseInt(dOutDate.getMonth())+1,
			dOutDate.getFullYear() );
	} else if ( sCalendarType == 'yearly' ) {
			//
			// The "day" is actually the selected year value.  The returned value will
			// be the first day of the selected year.
			//
		nSelectedYear = pnDay;
		prWinObj.opener.__returnField.value = ctCalendarFormatDate(
			prWinObj.opener.__dateFormat, 1, 1,
			nSelectedYear);
	} else {
			//
			// Returns the selected date for the standard, daily calendar type.
			//
		prWinObj.opener.__returnField.value = ctCalendarFormatDate(
			prWinObj.opener.__dateFormat, pnDay, nSelectedMonth + 1,
			nSelectedYear);
	}
		//
		// In case the opener window is no longer visible to the user
		// (because the user has mistakenly opened another window before
		// selecting a date via the calendar), return the focus to the
		// opener window before closing the calendar window.
		//
	this.focus();
	prWinObj.close();
}

/*
 * Builds the contents of the page that represents the calendar date
 * selection tool.  The calendar's contents are written into the
 * specified window object.
 *
 * Arguments:
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarShow(prWinObj) {
	var anDatePartsFromParseFunction = null;
		//
		// If the calendar popup is being requested,
		// determine the initial date to display in the calendar popup.
		// bErrorMessageDisplayed is included in the conditional to force
		// the calendar to display today's date following an error message.
		//
	if ( (bNewCalendar && prWinObj.opener.__returnField != null) ||
			bErrorMessageDisplayed) {
			//
			// If an error message was not previously displayed in the
			// calendar window, and if the value in the parent window's
			// return field is a valid date (regardless of the
			// delimiter characters that separate the month, day, year,
			// etc.), and if the return field's value is not an empty
			// string), set the initial date of the calendar to the
			// return field's value.
			// If an error message was displayed, or if the return field's
			// value is not valid (or is empty), set the initial date
			// of the calendar to today's date.
			//
		if (!bErrorMessageDisplayed &&
				isDate(prWinObj.opener.__returnField.value,	// Date
					prWinObj.opener.__dateFormat, 			// Date format
					"", 									// Error message
					true, 									// Delimiter ignore
					true, 									// Messages suppress
					false)) {								// Empty OK
				
			sStartingDate = prWinObj.opener.__returnField.value;
			anDatePartsFromParseFunction = ctParseDate(
					prWinObj.opener.__dateFormat, sStartingDate);
		} else {
	
			var rToday = new Date();
			var nTodayDay = rToday.getDate();
			var nTodayMonth = rToday.getMonth() + 1;
			var nTodayYear = rToday.getFullYear();
			var sTodayDay = "" + (nTodayDay < 10 ? "0" + nTodayDay : nTodayDay);
			var sTodayMonth =
					"" + (nTodayMonth < 10 ? "0" + nTodayMonth : nTodayMonth);
			var sTodayYear = "" + nTodayYear;
				//
				// Set the date parameters to match calendar type.  We
				// always need the beginning of the specified period.
				//
			if ( sCalendarType == 'monthly' || sCalendarType == 'quarterly' ) {
				sTodayDay = '01';
			} else if ( sCalendarType == 'yearly' ) {
				sTodayDay = '01';
				sTodayMonth = '01';
			}
				//
				// Set sStartingDate to today using a specific date format
				// and parse it using that specific format.
				//
			sStartingDate = "" + sTodayMonth + "/" + sTodayDay + "/" +
					sTodayYear;
			anDatePartsFromParseFunction = ctParseDate(
					"MM/dd/yyyy", sStartingDate);			
		}
			//
			// Create a date range based upon the selected day and calendar type.
			//
		ctCalendarSetDateRange();
		
		nSelectedYear = anDatePartsFromParseFunction[0];
		nSelectedMonth = anDatePartsFromParseFunction[1] - 1;
		nSelectedDay = anDatePartsFromParseFunction[2];
		rCurrDate.setFullYear(nSelectedYear, nSelectedMonth, nSelectedDay);
		nStartingYear = nSelectedYear;
		nStartingMonth = nSelectedMonth;
		nStartingDay = nSelectedDay;

		bErrorMessageDisplayed = false;
		bNewCalendar = false;
	}
		//
		// If the selected year is outside the range of years in the
		// year select box, notify the user.  Then set the
		// bErrorMessageDisplayed flag to true, so that the calendar
		// will display today's date when the button (created by the HTML
		// below) is pressed.
		//
	if(	nSelectedYear < CALENDAR_BASE_YEAR ||
			nSelectedYear > CALENDAR_MAX_YEAR	) {

		ctCalendarYearErrorMessage (prWinObj);

	} else {
			//
			// Generate the header of the page.
			//
		prWinObj.document.open();
		prWinObj.document.write('<html>');
		prWinObj.document.write('<head><title>Calendar</title>');
		prWinObj.document.write('<style>');
		prWinObj.document.write('A:link { text-decoration: none; color: black; }');
		prWinObj.document.write('A:hover { text-decoration: none; color: black; }');
		prWinObj.document.write('A:visited { text-decoration: none; color: black; }');
		prWinObj.document.write('</style></head>');
			//
			// Create the calendar body according to the type of calendar desired.
			//
		if ( sCalendarType == 'daily' || sCalendarType == 'weekly' ) {
			ctCalendarShowDaily(prWinObj);
		} else if ( sCalendarType == 'monthly' || sCalendarType == 'quarterly' ) {
			ctCalendarShowMonthly(prWinObj);
		} else if ( sCalendarType == 'yearly' ) {
			ctCalendarShowYearly(prWinObj);
		}
	}
	
}

/*
 * Builds the contents of the page that represents the calendar date
 * selection tool for a DAILY and WEEKLY calendar type.  The calendar's
 * contents are written into the specified window object.
 *
 * Arguments:
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarShowDaily(prWinObj) {
	var sDaysOfWeek = ["Sun","Mon","Tue","Wed","Thu","Fri","Sat"];
		//
		// Generate the Body of the page, including the form and tables
		// needed to contain the calendar.
		//
	prWinObj.document.write('<body bgcolor="white">');
	prWinObj.document.write('<form name="frmCalendar">');
	prWinObj.document.write('<table style="calendar" align="left" height="19" border="1" cellspacing="0" cellpadding="0">');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td colspan="7">');
	prWinObj.document.write('<table border="0" cellspacing="1" cellpadding="2" align="right">');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td align="right">&nbsp;</td>');
	prWinObj.document.write('<td align="right">');
		//
		// Create the calendar header which contains controls to select the month and
		// the year associated with the calendar tiles (days).
		//
	prWinObj.document.write('<select onChange="window.opener.ctCalendarChangeMonth(this, self)" name="cmbMonth" id="cmbMonth" size="1">');
	for ( nM = 0; nM < 12; nM++ ) {
		prWinObj.document.write('<option value="' + nM+1 + '"');
		if ( sMonthOfYear[nM] == sMonthOfYear[nSelectedMonth] ) {
			prWinObj.document.write(' selected');
		}
		prWinObj.document.write('>' + sMonthOfYear[nM] + '</option>');
	}
	prWinObj.document.write('</select>');
	prWinObj.document.write('</td>');

	prWinObj.document.write('<td align="right">');
	prWinObj.document.write('<select onChange="window.opener.ctCalendarChangeYear(this, self);" name="cmbYear" id="cmbYear" size="1">');
	for ( nYr = CALENDAR_BASE_YEAR; nYr <= CALENDAR_MAX_YEAR; nYr++ ) {
		prWinObj.document.write('<option value="' + nYr + '"');

		if ( nYr == nSelectedYear ) {

			prWinObj.document.write(' selected');
		}

		prWinObj.document.write('>' + nYr + '</option>');
	}
	prWinObj.document.write('</select>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('</table>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td>');
		//
		// Generate the table that contains the calendar's days.
		//
	prWinObj.document.write('<table border="0" cellspacing="1" cellpadding="0" align="center" bgcolor="silver">');
		//
		// Generate the weekday column headings.
		//
	prWinObj.document.write('<tr>');
	for ( nDays = 0; nDays < 7; nDays++ ) {
		prWinObj.document.write('<td height="20" width="24" bgcolor="#CCCCFF" align="center">');
		prWinObj.document.write('<span style="background-color: #CCCCFF; font-size: 8pt">' + sDaysOfWeek[nDays] + '</span>');
		prWinObj.document.write('</td>');
	}
	prWinObj.document.write('</tr>');
		//
		// Generate the first row of the calendar with blank tiles where there
		// is no date.
		//
	var nDayCounter = 0;
	rCurrDate.setDate(1);
	var nFirstDay = rCurrDate.getDay();
	prWinObj.document.write('<tr>');
	for ( nDay = 0; nDay < nFirstDay; nDay++ ) {
		prWinObj.document.write('<td height="24" width="24" bgcolor="Silver">');
		prWinObj.document.write('<IMG src="../CeleritasCommon/images/transparent1X1.gif" height="24" width="24" border="0">');
		prWinObj.document.write('</td>');
	}
	for ( nDay = nFirstDay; nDay < 7; nDay++ ) {
		nDayCounter++;
		prWinObj.document.write('<td height="24" width="24" align="center" bgcolor="#');
		prWinObj.document.write(ctCalendarGetDayColor(nSelectedYear, nSelectedMonth, nDayCounter));
		prWinObj.document.write('">');
		prWinObj.document.write('<a href="#" onclick="window.opener.ctCalendarSelectDay(' + nDayCounter + ', self);"');
		prWinObj.document.write('<span name="txtDay_' + nDayCounter + '" style="font-size: 8pt">' + nDayCounter + '</span>');
		prWinObj.document.write('</a></td>');
	}
	prWinObj.document.write('</tr>');
		//
		// Depending upon the starting day of the month and the number of days in the
		// month, the calendar may need six rows rather than five to contain all
		// of the days in the proper sequence.  The code determines if an extra (sixth)
		// row is needed.
		//
	var nLastDay = ctCalendarGetDaysOfMonth(nSelectedMonth, nCurrYear);
	var nWeeksInMiddle = 4;
	if ( (35 - nLastDay - nFirstDay) < 0 ) {
		nWeeksInMiddle = 5;
	}
		//
		// Generate the center portion of the calendar.
		//
	for ( nWeek = 1; nWeek < nWeeksInMiddle; nWeek++ ) {
		prWinObj.document.write('<tr>');
		for ( nDay = 0; nDay < 7; nDay++ ) {
			nDayCounter++;
			prWinObj.document.write('<td height="24" width="24" align="center" bgcolor="#');
			prWinObj.document.write(ctCalendarGetDayColor(nSelectedYear, nSelectedMonth, nDayCounter));
			prWinObj.document.write('">');
			prWinObj.document.write('<a href="#" onclick="window.opener.ctCalendarSelectDay(' + nDayCounter + ', self);"');
			prWinObj.document.write('<span name="txtDay_' + nDayCounter + '" style="font-size: 8pt">' + nDayCounter + '</span>');
			prWinObj.document.write('</a></td>');
		}
		prWinObj.document.write('</tr>');
	}
		//
		// Generate the last row of the calendar with blank tiles where there
		// is no date.
		//
	prWinObj.document.write('<tr>');
	for ( nDay = nDayCounter; nDay < nLastDay; nDay++ ) {
		nDayCounter++;
		prWinObj.document.write('<td height="24" width="24" align="center" bgcolor="#');
		prWinObj.document.write(ctCalendarGetDayColor(nSelectedYear, nSelectedMonth, nDayCounter));
		prWinObj.document.write('">');
		prWinObj.document.write('<a href="#" onclick="window.opener.ctCalendarSelectDay(' + nDayCounter + ', self);"');
		prWinObj.document.write('<span style="font-size: 8pt">' + nDayCounter + '</span>');
		prWinObj.document.write('</a></td>');
	}
	for ( nDay = nLastDay; nDay < 35 - nFirstDay; nDay++ ) {
		prWinObj.document.write('<td height="24" width="24" bgcolor="Silver">');
		prWinObj.document.write('<IMG src="../CeleritasCommon/images/transparent1X1.gif" height="24" width="24" border="0">');
		prWinObj.document.write('</td>');
	}
	prWinObj.document.write('</tr>');

	prWinObj.document.write('</table>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('</table>');
	prWinObj.document.write('</form>');
	prWinObj.document.write('</body>');
	prWinObj.document.close();
}

/*
 * Builds the contents of the page that represents the calendar date
 * selection tool for a MONTHLY and QUARTERLY calendar type.  The 
 * calendar's contents are written into the specified window object.
 *
 * Arguments:
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarShowMonthly(prWinObj) {
	var sMonthsOfYear = ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"];
		//
		// Generate the Body of the page, including the form and tables
		// needed to contain the calendar.
		//
	prWinObj.document.write('<body bgcolor="white">');
	prWinObj.document.write('<form name="frmCalendar">');
	prWinObj.document.write('<table style="calendar" align="center" height="19" border="1" cellspacing="0" cellpadding="0">');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td colspan="7" bgcolor="#CCCCFF">');
	prWinObj.document.write('<table border="0" cellspacing="1" cellpadding="2" align="center">');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td align="center">');
		//
		// Create the calendar header which contains a control to select the year
		// associated with the calendar tiles (months).
		//
	prWinObj.document.write('Year: </td>');
	prWinObj.document.write('<td align="center">');
	prWinObj.document.write('<select onChange="window.opener.ctCalendarChangeYear(this, self);" name="cmbYear" id="cmbYear" size="1">');
	for ( nYr = CALENDAR_BASE_YEAR; nYr <= CALENDAR_MAX_YEAR; nYr++ ) {
		prWinObj.document.write('<option value="' + nYr + '"');

		if ( nYr == nSelectedYear ) {

			prWinObj.document.write(' selected');
		}

		prWinObj.document.write('>' + nYr + '</option>');
	}
	prWinObj.document.write('</select>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('</table>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td>');
		//
		// Generate the table that contains the calendar's months.
		//
	prWinObj.document.write('<table border="0" cellspacing="1" cellpadding="0" align="center" bgcolor="silver">');
		//
		// Generate the center portion of the calendar.  We will have four rows
		// of three months.
		//
	var nMonthCounter = 0;
	rCurrDate.setDate(1);
	for ( nRow = 0; nRow < 4 ; nRow++ ) {
		prWinObj.document.write('<tr>');
		for ( nCol = 0; nCol < 3; nCol++ ) {
			nMonthCounter++;
			prWinObj.document.write('<td height="42" width="42" align="center" bgcolor="#');
			prWinObj.document.write(ctCalendarGetDayColor( nSelectedYear, nMonthCounter-1, 1 ));
			prWinObj.document.write('">');
			prWinObj.document.write('<a href="#" onclick="window.opener.ctCalendarSelectDay(' + nMonthCounter + ', self);"');
			prWinObj.document.write('<span name="txtDay_' + nMonthCounter + '" style="font-size: 10pt">' 
									+ sMonthsOfYear[nMonthCounter-1] + '</span>');
			prWinObj.document.write('</a></td>');
		}
		prWinObj.document.write('</tr>');
	}

	prWinObj.document.write('</table>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('</table>');
	prWinObj.document.write('</form>');
	prWinObj.document.write('</body>');
	prWinObj.document.close();
	
}

/*
 * Builds the contents of the page that represents the calendar date
 * selection tool for a YEARLY calendar type.  The calendar's contents
 * are written into the specified window object.
 *
 * Arguments:
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarShowYearly(prWinObj) {
		//
		// Generate the Body of the page, including the form and tables
		// needed to contain the calendar.
		//
	prWinObj.document.write('<body bgcolor="white">');
	prWinObj.document.write('<form name="frmCalendar">');
	prWinObj.document.write('<table style="calendar" align="center" height="19" border="1" cellspacing="0" cellpadding="0">');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td colspan="5" bgcolor="#CCCCFF">');
		//
		// Create the calendar header.
		//
	prWinObj.document.write('<table border="0" cellspacing="0" cellpadding="2" align="center">');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td align="left">');
	prWinObj.document.write('<a href="javascript:window.opener.ctCalendarBumpYearlyDisplay(-5, self);">');
	prWinObj.document.write('&laquo;</a></td>');
	prWinObj.document.write('<td>&nbsp;</td>');
	prWinObj.document.write('<td align="center">');
	prWinObj.document.write('Select A Year: </td>');
	prWinObj.document.write('<td>&nbsp;</td>');
	prWinObj.document.write('<td align="right">');
	prWinObj.document.write('<a href="javascript:window.opener.ctCalendarBumpYearlyDisplay(5, self);">');
	prWinObj.document.write('&raquo;</a></td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('</table>');

	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('<tr>');
	prWinObj.document.write('<td>');
		//
		// Generate the table that contains the calendar's years.
		//
	prWinObj.document.write('<table border="0" cellspacing="1" cellpadding="0" align="center" bgcolor="silver">');
		//
		// Generate the center portion of the calendar.  We will have three rows of
		// five years.  Initially display up to 10 years before and 5 years after the
		// currrent date.  The display can be adjusted by 5 years using the arrows
		// at the top of the calendar.
		//
	var dNow = new Date();
	var nCurrentYear = dNow.getFullYear() - 9 + parseInt(nYearlyDisplayIncrement);
	if ( nCurrentYear < CALENDAR_BASE_YEAR ) {
		nCurrentYear = CALENDAR_BASE_YEAR;
	}
	rCurrDate.setDate(1);
	for ( nRow = 0; nRow < 3 ; nRow++ ) {
		prWinObj.document.write('<tr>');
		var nCol = 0;
		while ( nCol < 5 && nCurrentYear <= CALENDAR_MAX_YEAR ) {
			prWinObj.document.write('<td height="30" width="30" align="center" bgcolor="#');
			prWinObj.document.write(ctCalendarGetDayColor( nCurrentYear, 0, 1 ));
			prWinObj.document.write('">');
			prWinObj.document.write('<a href="#" onclick="window.opener.ctCalendarSelectDay(' + nCurrentYear + ', self);"');
			prWinObj.document.write('<span name="txtDay_' + nCurrentYear + '" style="font-size: 8pt">' 
									+ nCurrentYear + '</span>');
			prWinObj.document.write('</a></td>');
			nCurrentYear++;
			nCol++;
		}
		prWinObj.document.write('</tr>');
	}

	prWinObj.document.write('</table>');
	prWinObj.document.write('</td>');
	prWinObj.document.write('</tr>');
	prWinObj.document.write('</table>');
	prWinObj.document.write('</form>');
	prWinObj.document.write('</body>');
	prWinObj.document.close();
	
}

/*
 * Creates a date range based upon the selected starting date and the
 * calendar type.
 *
 * Arguments:
 * 		none
 *
 * Returns:
 *		Void
 */
function ctCalendarSetDateRange() {
	var dStartDate = new Date(sStartingDate);
	var dEndDate = new Date(sStartingDate);
	var nWeekDay = dStartDate.getDay();

	dEndDate.setHours(23);
	dEndDate.setMinutes(59);
	dEndDate.setSeconds(59);
	dEndDate.setMilliseconds(999);

	if ( sCalendarType == 'daily' ) {
			//
			// Range is the selected day.
			//
		nDateRangeStart = dStartDate.getTime();
		nDateRangeEnd = dEndDate.getTime();
	} else if (sCalendarType == 'weekly' ) {
			//
			// Range is the first day of the week to the last day of the week.
			// (Note:  Javascript week days are numbered 0 to 6.)
			//
		nDateRangeStart = dStartDate.getTime() - nWeekDay*MILLIS_PER_DAY;
		nDateRangeEnd = dEndDate.getTime() + (6 - nWeekDay)*MILLIS_PER_DAY;
	} else if ( sCalendarType == 'monthly' ) {
			//
			// Range is the first day of the month to the last day of the month.
			//
		dStartDate.setDate(1);
		dEndDate.setDate(nLeapDaysOfMonth[dStartDate.getMonth()]);

		nDateRangeStart = dStartDate.getTime();
		nDateRangeEnd = dEndDate.getTime();
	} else if ( sCalendarType == 'quarterly' ) {
			//
			// Range is the first day of the quarter to the last day of the quarter.
			//
		var nStartMonth = dStartDate.getMonth();
		var nEndMonth = nStartMonth;
		if ( nStartMonth % 3 == 0 ) {
			//
			// In the first month of the quarter.
			//
		} else if ( nStartMonth % 3 == 1 ) {
			//
			// In the second month of the quarter.
			//
			nStartMonth = nStartMonth - 1;
			dStartDate.setMonth(nStartMonth);
		} else if ( nStartMonth % 3 == 2 ) {
			//
			// In the third month of the quarter.
			//
			nStartMonth = nStartMonth - 2;
			dStartDate.setMonth(nStartMonth);
		}
		dStartDate.setDate(1);
		nEndMonth = nStartMonth + 2;
		dEndDate.setMonth(nEndMonth);
		dEndDate.setDate(nLeapDaysOfMonth[nEndMonth]);

		nDateRangeStart = dStartDate.getTime();
		nDateRangeEnd = dEndDate.getTime();
	} else if ( sCalendarType == 'yearly' ) {
			//
			// Range is the first day of the year to the last day of the year.
			// (Note:  Javascript months are numbered 0 to 11.)
			//
		dStartDate.setMonth(0);
		dStartDate.setDate(1);

		dEndDate.setMonth(11);
		dEndDate.setDate(31);
		
		nDateRangeStart = dStartDate.getTime();
		nDateRangeEnd = dEndDate.getTime();
	}
}

/*
 * Increments a variable which controls the offset of the yearly calendar display
 * from the original display whenever the shift arrows are clicked at the top
 * of the calendar.  Once the variable is incremented, the calendar is updated
 * to display the new set of yearly selections available.
 *
 * Arguments:
 *		pnIncrement - The number of years to increment the yearly display.
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarBumpYearlyDisplay( pnIncrement, prWinObj ) {
	nYearlyDisplayIncrement += pnIncrement;
		//
		// Don't let the calendar shift more than 5 years back
		// from its original display.
		//
	if ( nYearlyDisplayIncrement < -5 ) {
		nYearlyDisplayIncrement = -5;
	}
	ctCalendarUpdateCalendar(nSelectedMonth, nSelectedYear, prWinObj);
}

/*
 * Displays an error message if the selected year is outside of the
 * calendar's default range.
 *
 * Arguments:
 * 		prWinObj	- The window object which contains the calendar.
 *
 * Returns:
 *		Void
 */
function ctCalendarYearErrorMessage (prWinObj) {
		prWinObj.document.open();
		prWinObj.document.write('<html>');
		prWinObj.document.write('<head><title>Calendar Error</title>');
		prWinObj.document.write('<link rel="stylesheet" type="text/css" ' +
				'href="./css/so-standard.css">');
		prWinObj.document.write('</head>');
		
		prWinObj.document.write('<body bgcolor="white">');
		prWinObj.document.write('<form name="frmCalendarError">');
		prWinObj.document.write('<table align="left" border="0" ' +
				'cellspacing="0" cellpadding="0">');
		prWinObj.document.write('<tr>');
		prWinObj.document.write('<td>');
		prWinObj.document.write('<IMG src="../CeleritasCommon/images/transparent1X1.gif" '+
				'height="15" border="0">');
		prWinObj.document.write('</td>');
		prWinObj.document.write('</tr>');
		prWinObj.document.write('<tr>');
		prWinObj.document.write('<td class="ErrorMessage">');
		prWinObj.document.write('The selected year is not able to be ' +
				'displayed on the calendar.&nbsp&nbspThe calendar can ' +
				'display years from ' +	CALENDAR_BASE_YEAR + ' to ' +
				CALENDAR_MAX_YEAR + '.');
		prWinObj.document.write('</td>');
		prWinObj.document.write('</tr>');

		prWinObj.document.write('<tr>');
		prWinObj.document.write('<td>');
		prWinObj.document.write('<IMG src="../CeleritasCommon/images/transparent1X1.gif" ' +
				'height="20" border="0">');
		prWinObj.document.write('</td>');
		prWinObj.document.write('</tr>');

		prWinObj.document.write('<tr>');
		prWinObj.document.write('<td align="center">');
		prWinObj.document.write('<input type="button" ' +
				'name="CalendarErrorBtn" value="Show Calendar" ' +
				'onClick="window.opener.ctCalendarShow(self)">');
		prWinObj.document.write('</td>');
		prWinObj.document.write('</tr>')
		prWinObj.document.write('</table>');
		prWinObj.document.write('</form>');
		prWinObj.document.write('</body>');
		prWinObj.document.close();

		bErrorMessageDisplayed = true;
		bNewCalendar = true;
}

/*
 * Creates a pop-up window and generates a date selector calendar within 
 * the window.  When a user selects a date from the calendar the calendar
 * window updates a specified date input field in the source window.
 *
 * Arguments:
 * 		sDateFormat		- The date format to use when storing a date into the 
 *						  source input field.
 *		rReturnField	- An object reference to the target date input field.
 *		sType			- The type of calendar to display.  Valid types are daily,
 *						  weekly, monthly, quarterly, and yearly.  Daily is the
 *						  default type if this parameter is not supplied.
 *
 * Returns:
 *		Void
 */
function ctCalendarPopUp(sDateFormat, rReturnField, sType) {

	rCurrDate = new Date();
	
	if ( sType != null && sType != "" ) {
		sCalendarType = sType;
	}

	rWinCal = showPopUp("", // Page content
				"Calendar",	// Page title
				300, 		// Left offset
				200,		// Top offset
				200, 		// Width 
				225, 		// Height
				false,		// Resizable
				false,		// Scrollable
				false,		// Status bar
				false );	// Modal
	
	rWinCal.opener = self;
	rWinCal.opener.__dateFormat = sDateFormat;
	rWinCal.opener.__returnField = rReturnField;
	rWinCal.focus();
	
	bErrorMessageDisplayed = false;
	bNewCalendar = true;

	ctCalendarShow(rWinCal);
}


